import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { motion } from 'framer-motion'
import { Calendar, Code, CheckCircle, ArrowRight, Zap, Layers, Target, Award, Users, Clock, TrendingUp, Sparkles, Star } from 'lucide-react'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedProgram, setSelectedProgram] = useState('')
  const [startDate, setStartDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const programs = [
    { 
      id: 'blender', 
      name: 'Blender', 
      icon: Code,
      color: '#F57900',
      gradient: 'linear-gradient(135deg, #F57900 0%, #FF9500 100%)',
      description: 'Бесплатное и мощное ПО для 3D-моделирования',
      duration: '8 недель',
      level: 'Начинающий',
      students: '2500+'
    },
    { 
      id: 'maya', 
      name: 'Maya', 
      icon: Layers,
      color: '#00D9FF',
      gradient: 'linear-gradient(135deg, #00D9FF 0%, #7C3AED 100%)',
      description: 'Профессиональный стандарт индустрии',
      duration: '10 недель',
      level: 'Продвинутый',
      students: '1800+'
    },
    { 
      id: '3dsmax', 
      name: '3ds Max', 
      icon: Target,
      color: '#7C3AED',
      gradient: 'linear-gradient(135deg, #7C3AED 0%, #F59E0B 100%)',
      description: 'Мощный инструмент для архитектуры и визуализации',
      duration: '12 недель',
      level: 'Профессионал',
      students: '3200+'
    },
    { 
      id: 'zbrush', 
      name: 'ZBrush', 
      icon: Sparkles,
      color: '#F59E0B',
      gradient: 'linear-gradient(135deg, #F59E0B 0%, #00D9FF 100%)',
      description: 'Цифровая скульптура и детализация',
      duration: '9 недель',
      level: 'Продвинутый',
      students: '2100+'
    }
  ]

  const instructors = [
    { name: 'Алексей Волков', program: 'Blender', experience: '12 лет', rating: 5, students: '2500+' },
    { name: 'Мария Соколова', program: 'Maya', experience: '15 лет', rating: 5, students: '1800+' },
    { name: 'Дмитрий Новиков', program: '3ds Max', experience: '10 лет', rating: 5, students: '3200+' },
    { name: 'Елена Петрова', program: 'ZBrush', experience: '8 лет', rating: 5, students: '2100+' }
  ]

  const processSteps = [
    { icon: Code, title: 'Выберите программу', description: 'Выберите подходящий курс для вашего уровня', step: '01' },
    { icon: Calendar, title: 'Укажите дату', description: 'Выберите удобную дату начала обучения', step: '02' },
    { icon: CheckCircle, title: 'Подтвердите', description: 'Оформите заявку и получите доступ', step: '03' },
    { icon: Zap, title: 'Начните обучение', description: 'Приступайте к созданию 3D-моделей', step: '04' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedProgram && startDate) {
      const savedProjects = localStorage.getItem('model3dProjects')
      const projects = savedProjects ? JSON.parse(savedProjects) : []
      
      const programData = programs.find(p => p.id === selectedProgram)
      const instructor = instructors.find(i => i.program === programData?.name)
      
      const newProject = {
        id: Date.now(),
        course: programData?.name || 'Курс',
        instructor: instructor?.name || 'Преподаватель',
        submissionDate: startDate,
        rating: 0,
        progress: 0
      }

      projects.push(newProject)
      localStorage.setItem('model3dProjects', JSON.stringify(projects))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedProgram('')
        setStartDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="enroll">
      <div className="container">
        <motion.section 
          className="enroll-hero-tech"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8 }}
        >
          <div className="enroll-hero-tech-grid">
            <motion.div 
              className="enroll-hero-tech-main"
              initial={{ opacity: 0, x: -60 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.8, delay: 0.2 }}
            >
              <div className="enroll-tech-badge">
                <Zap size={20} strokeWidth={2.5} />
                <span>ЗАПИСАТЬСЯ НА КУРС</span>
              </div>
              <h1 className="enroll-tech-title">
                Выберите <span className="enroll-tech-accent">программу</span>
              </h1>
              <div className="enroll-tech-line"></div>
            </motion.div>
            <motion.div 
              className="enroll-hero-tech-side"
              initial={{ opacity: 0, x: 60 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.8, delay: 0.3 }}
            >
              <p className="enroll-tech-description">
                Начните свой путь в 3D-моделировании. 
                Выберите программу и дату начала обучения.
              </p>
            </motion.div>
          </div>
        </motion.section>

        <motion.section 
          className="enroll-form-tech"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.7, delay: 0.4 }}
        >
          <div className="enroll-form-tech-wrapper">
            <form className="enroll-form-tech-inner" onSubmit={handleSubmit}>
              <div className="form-tech-section">
                <div className="form-tech-label">
                  <Code size={32} strokeWidth={2.5} className="label-tech-icon" />
                  <span className="label-tech-text">Программа обучения</span>
                </div>
                <div className="programs-tech-grid">
                  {programs.map((program, index) => {
                    const IconComponent = program.icon
                    return (
                      <motion.button
                        key={program.id}
                        type="button"
                        className={`program-tech-card ${selectedProgram === program.id ? 'selected' : ''}`}
                        onClick={() => setSelectedProgram(program.id)}
                        initial={{ opacity: 0, y: 40, rotateX: -15 }}
                        animate={{ opacity: 1, y: 0, rotateX: 0 }}
                        transition={{ delay: 0.5 + index * 0.1, duration: 0.6 }}
                        whileHover={{ 
                          y: -10, 
                          rotateY: 5,
                          scale: 1.03,
                          transition: { duration: 0.3 } 
                        }}
                        style={{ 
                          background: selectedProgram === program.id ? program.gradient : 'rgba(26, 31, 58, 0.6)',
                          borderColor: selectedProgram === program.id ? program.color : 'rgba(0, 217, 255, 0.2)'
                        }}
                      >
                        <div className="program-tech-icon-wrapper" style={{ 
                          background: selectedProgram === program.id ? 'rgba(255, 255, 255, 0.2)' : 'rgba(0, 217, 255, 0.1)',
                          borderColor: selectedProgram === program.id ? 'rgba(255, 255, 255, 0.4)' : 'rgba(0, 217, 255, 0.3)'
                        }}>
                          <IconComponent size={48} strokeWidth={2.5} color={selectedProgram === program.id ? 'white' : program.color} />
                          <div className="program-tech-icon-glow" style={{ opacity: selectedProgram === program.id ? 1 : 0.5 }}></div>
                        </div>
                        <div className="program-tech-content">
                          <h3 className="program-tech-name" style={{ color: selectedProgram === program.id ? 'white' : 'var(--text)' }}>
                            {program.name}
                          </h3>
                          <p className="program-tech-description" style={{ color: selectedProgram === program.id ? 'rgba(255, 255, 255, 0.9)' : 'var(--text-light)' }}>
                            {program.description}
                          </p>
                          <div className="program-tech-meta">
                            <div className="program-tech-meta-item">
                              <Clock size={16} strokeWidth={2} />
                              <span>{program.duration}</span>
                            </div>
                            <div className="program-tech-meta-item">
                              <TrendingUp size={16} strokeWidth={2} />
                              <span>{program.level}</span>
                            </div>
                            <div className="program-tech-meta-item">
                              <Users size={16} strokeWidth={2} />
                              <span>{program.students}</span>
                            </div>
                          </div>
                        </div>
                        {selectedProgram === program.id && (
                          <motion.div 
                            className="program-tech-check"
                            initial={{ scale: 0, rotate: -180 }}
                            animate={{ scale: 1, rotate: 0 }}
                            transition={{ type: "spring", stiffness: 200, damping: 15 }}
                          >
                            <CheckCircle size={36} strokeWidth={2.5} color="white" />
                          </motion.div>
                        )}
                        <div className="program-tech-glow" style={{ opacity: selectedProgram === program.id ? 1 : 0 }}></div>
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              {selectedProgram && (
                <motion.div 
                  className="form-tech-section date-tech-section"
                  initial={{ opacity: 0, height: 0 }}
                  animate={{ opacity: 1, height: 'auto' }}
                  transition={{ duration: 0.6 }}
                >
                  <div className="date-tech-label">
                    <Calendar size={36} strokeWidth={2.5} className="label-tech-icon" />
                    <div className="date-tech-label-content">
                      <span className="date-tech-label-title">Дата начала обучения</span>
                      <span className="date-tech-label-hint">Можно записаться на ближайшие 90 дней</span>
                    </div>
                  </div>
                  <div className="date-tech-input-wrapper">
                    <input
                      type="date"
                      id="startDate"
                      className="form-tech-input"
                      value={startDate}
                      onChange={(e) => setStartDate(e.target.value)}
                      min={minDate}
                      max={maxDate}
                      required
                    />
                    <div className="date-tech-input-glow"></div>
                  </div>
                </motion.div>
              )}

              <motion.button 
                type="submit" 
                className="submit-tech-button" 
                disabled={!selectedProgram || !startDate}
                whileHover={{ scale: 1.05, y: -4, rotateX: 5 }}
                whileTap={{ scale: 0.95 }}
              >
                <span>Записаться на курс</span>
                <ArrowRight size={32} strokeWidth={2.5} className="btn-tech-icon" />
              </motion.button>

              {isSubmitted && (
                <motion.div 
                  className="success-tech-message"
                  initial={{ opacity: 0, scale: 0.8, rotateX: -15 }}
                  animate={{ opacity: 1, scale: 1, rotateX: 0 }}
                  transition={{ duration: 0.5, type: "spring" }}
                >
                  <CheckCircle size={48} strokeWidth={2.5} className="success-tech-icon" />
                  <span>Вы успешно записаны на курс!</span>
                </motion.div>
              )}
            </form>
          </div>
        </motion.section>

        <motion.section 
          className="instructors-tech"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.7, delay: 0.5 }}
        >
          <div className="instructors-tech-header">
            <div className="section-tech-line"></div>
            <h2 className="section-tech-title">Наши преподаватели</h2>
          </div>
          <div className="instructors-tech-grid">
            {instructors.map((instructor, index) => (
              <motion.div
                key={instructor.name}
                className="instructor-tech-card"
                initial={{ opacity: 0, y: 40, rotateX: -15 }}
                animate={{ opacity: 1, y: 0, rotateX: 0 }}
                transition={{ delay: 0.6 + index * 0.1, duration: 0.6 }}
                whileHover={{ 
                  y: -10, 
                  rotateY: 5,
                  scale: 1.03,
                  transition: { duration: 0.3 } 
                }}
              >
                <div className="instructor-tech-avatar" style={{ 
                  background: index % 4 === 0 ? 'linear-gradient(135deg, #F57900 0%, #FF9500 100%)' :
                  index % 4 === 1 ? 'linear-gradient(135deg, #00D9FF 0%, #7C3AED 100%)' :
                  index % 4 === 2 ? 'linear-gradient(135deg, #7C3AED 0%, #F59E0B 100%)' :
                  'linear-gradient(135deg, #F59E0B 0%, #00D9FF 100%)'
                }}>
                  <Users size={40} strokeWidth={2.5} color="white" />
                </div>
                <h3 className="instructor-tech-name">{instructor.name}</h3>
                <div className="instructor-tech-program">{instructor.program}</div>
                <div className="instructor-tech-details">
                  <div className="instructor-tech-detail">
                    <Clock size={16} strokeWidth={2} />
                    <span>{instructor.experience}</span>
                  </div>
                  <div className="instructor-tech-detail">
                    <Star size={16} strokeWidth={2} fill="#F59E0B" color="#F59E0B" />
                    <span>{instructor.rating}.0</span>
                  </div>
                  <div className="instructor-tech-detail">
                    <Users size={16} strokeWidth={2} />
                    <span>{instructor.students}</span>
                  </div>
                </div>
                <div className="instructor-tech-glow"></div>
              </motion.div>
            ))}
          </div>
        </motion.section>

        <motion.section 
          className="process-tech"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.7, delay: 0.7 }}
        >
          <div className="process-tech-header">
            <div className="section-tech-line"></div>
            <h2 className="section-tech-title">Как это работает</h2>
          </div>
          <div className="process-tech-grid">
            {processSteps.map((step, index) => {
              const IconComponent = step.icon
              return (
                <motion.div
                  key={step.title}
                  className="process-tech-step"
                  initial={{ opacity: 0, y: 40, rotateX: -15 }}
                  animate={{ opacity: 1, y: 0, rotateX: 0 }}
                  transition={{ delay: 0.8 + index * 0.1, duration: 0.6 }}
                  whileHover={{ 
                    y: -10, 
                    rotateY: 5,
                    scale: 1.05,
                    transition: { duration: 0.3 } 
                  }}
                >
                  <div className="process-tech-step-number">{step.step}</div>
                  <div className="process-tech-step-icon" style={{ 
                    background: index % 4 === 0 ? 'linear-gradient(135deg, #00D9FF 0%, #7C3AED 100%)' :
                    index % 4 === 1 ? 'linear-gradient(135deg, #7C3AED 0%, #F59E0B 100%)' :
                    index % 4 === 2 ? 'linear-gradient(135deg, #00D9FF 0%, #F59E0B 100%)' :
                    'linear-gradient(135deg, #F59E0B 0%, #7C3AED 100%)'
                  }}>
                    <IconComponent size={48} strokeWidth={2.5} color="white" />
                    <div className="process-tech-step-icon-glow"></div>
                  </div>
                  <h3 className="process-tech-step-title">{step.title}</h3>
                  <p className="process-tech-step-description">{step.description}</p>
                </motion.div>
              )
            })}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default Enroll

